const express = require('express');
const path = require('path');
const bodyParser = require('body-parser');
const fs = require('fs');
const axios = require('axios');
const https = require('https');
const url = require('url');

const app = express();
const PORT = process.env.PORT || 3000;

// Middleware
app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ extended: true }));
app.use(express.static(path.join(__dirname, 'public')));


const configPath = path.join(__dirname, 'config.json');
if (!fs.existsSync(configPath)) {
  fs.writeFileSync(configPath, JSON.stringify({
    apiUrl: 'http://localhost:8080'
  }, null, 2));
}

let config = JSON.parse(fs.readFileSync(configPath, 'utf8'));

function generateUniqueTransactionSequence() {
  const timestamp = Date.now();
  const random = Math.floor(Math.random() * 1000);
  return Math.abs(timestamp + random) % 1000000000;
}

let lastTransactionSequence = generateUniqueTransactionSequence();

app.get('/', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'index.html'));
});

app.get('/api/settings', (req, res) => {
  res.json(config);
});

app.post('/api/settings', (req, res) => {
  config = { ...config, ...req.body };
  fs.writeFileSync(configPath, JSON.stringify(config, null, 2));
  res.json({ success: true, message: 'Ayarlar güncellendi', config });
});

app.post('/api/pairing', async (req, res) => {
  try {
    const { deviceSerial } = req.body;

    const transactionSequence = generateUniqueTransactionSequence();
    lastTransactionSequence = transactionSequence;

    const serialNumber = deviceSerial || "N860W6A2550";

    const response = await axios.post(`${config.apiUrl}/Pairing`, {
      TransactionHandle: {
        SerialNumber: serialNumber,
        TransactionDate: new Date().toISOString(),
        TransactionSequence: transactionSequence,
        Fingerprint: "test1"
      }
    });

    console.log(`Pairing işlemi: Cihaz SN: ${serialNumber}, İşlem Numarası: ${transactionSequence}`);
    res.json(response.data);
  } catch (error) {
    console.error('Eşleşme hatası:', error.message);
    res.status(500).json({
      HasError: true,
      Message: `Eşleşme hatası: ${error.message}`
    });
  }
});

app.post('/api/complete-sale', async (req, res) => {
  try {
    const { cart, paymentMethod, deviceSerial, deviceIp } = req.body;

    if (!cart || cart.length === 0) {
      return res.status(400).json({
        HasError: true,
        Message: "Sepet boş, satış tamamlanamaz"
      });
    }

    if (!deviceSerial) {
      return res.status(400).json({
        HasError: true,
        Message: "Cihaz seri numarası belirtilmemiş"
      });
    }

    const transactionSequence = generateUniqueTransactionSequence();

    if (transactionSequence === lastTransactionSequence) {
      transactionSequence = generateUniqueTransactionSequence();
    }

    lastTransactionSequence = transactionSequence;

    console.log(`CompleteSale işlemi: Cihaz SN: ${deviceSerial}, IP: ${deviceIp || 'Belirtilmemiş'}, İşlem Numarası: ${transactionSequence}`);

    const totalPrice = cart.reduce((total, item) => total + (item.price * item.quantity), 0);

    const saleItems = cart.map(item => ({
      Name: item.name,
      IsGeneric: false,
      UnitCode: "KGM",
      TaxGroupCode: "KDV18",
      ItemQuantity: item.quantity,
      UnitPriceAmount: item.price,
      GrossPriceAmount: item.price * item.quantity,
      TotalPriceAmount: item.price * item.quantity,
      ReservedText: "TEST0001"
    }));

    let saleData = {
      TransactionHandle: {
        SerialNumber: deviceSerial,
        TransactionDate: new Date().toISOString(),
        TransactionSequence: transactionSequence,
        Fingerprint: "test1"
      },
      Sale: {
        RefererApp: "Ytd POS",
        RefererAppVersion: "1.0.0",
        OrderNo: `ORDER${Date.now().toString().substring(8)}`, // sipariş numarası
        MainDocumentType: 1,
        GrossPrice: totalPrice,
        TotalPrice: totalPrice,
        CurrencyCode: "TRY",
        ExchangeRate: 1,
        AddedSaleItems: saleItems,
        ReceiptInformation: {
          ReceiptImageEnabled: true,
          ReceiptWidth: "58mm",
          PrintCustomerReceipt: true,
          PrintCustomerReceiptCopy: true,
          PrintMerchantReceipt: true,
          EnableExchangeRateField: true
        }
      }
    };

    if (paymentMethod === 'cash') {
      // Nakit ödeme
      saleData.Sale.PaymentInformations = [{
        Mediator: 2, // Nakit ödeme için
        Amount: totalPrice,
        CurrencyCode: "TRY",
        ExchangeRate: 1
      }];
      saleData.Sale.ExternalPayments = [];
    } else {
      // Kredi kartı ödeme 
      saleData.Sale.PaymentInformations = [];
      saleData.Sale.ExternalPayments = [{
        Type: 2,
        Mediator: 9, // Kredi kartı için
        Brand: 9,
        CardNo: "554960******2794",
        AuthorizationCode: "155",
        Amount: totalPrice,
        CurrencyCode: "TRY",
        ExchangeRate: 1
      }];
    }

    console.log('Satış isteği gönderiliyor...');
    console.log('Toplam fiyat:', totalPrice);

    let apiUrl = config.apiUrl;

    if (deviceIp && deviceIp.trim() !== '') {
      let deviceApiUrl = deviceIp.trim();
      if (!deviceApiUrl.startsWith('http://') && !deviceApiUrl.startsWith('https://')) {
        deviceApiUrl = `http://${deviceApiUrl}`;
      }

      const parsedUrl = new URL(config.apiUrl);
      const port = deviceApiUrl.includes(':') ? '' : parsedUrl.port ? `:${parsedUrl.port}` : '';
      const path = parsedUrl.pathname !== '/' ? parsedUrl.pathname : '';

      if (port && !deviceApiUrl.includes(port)) {
        apiUrl = `${deviceApiUrl}${port}${path}`;
      } else {
        apiUrl = `${deviceApiUrl}${path}`;
      }

      console.log(`Cihazın IP adresi belirtildi, API URL güncellendi: ${apiUrl}`);
    }

    const httpsAgent = new https.Agent({
      rejectUnauthorized: false
    });

    const axiosConfig = {
      method: 'post',
      url: `${apiUrl}/CompleteSale`,
      headers: {
        'Connection': 'close',
        'Content-Type': 'application/json'
      },
      httpsAgent,
      data: saleData,
      timeout: 15000
    };

    const response = await axios(axiosConfig);

    console.log('Satış yanıtı alındı:', response.status);
    res.json(response.data);
  } catch (error) {
    console.error('HATA DETAYI:', error);

    let errorMessage = error.message;
    let errorDetail = {};

    if (error.response) {
      errorMessage = `Sunucu yanıtı: ${error.response.status}`;
      errorDetail = error.response.data;
      console.error('YANIT VERİLERİ:', JSON.stringify(error.response.data, null, 2));
      console.error('YANIT DURUM KODU:', error.response.status);
    } else if (error.request) {
      errorMessage = 'Sunucudan yanıt alınamadı';
      errorDetail = { request: 'Yanıt alınamadı' };
    }

    console.error('Satış tamamlama hatası:', errorMessage);
    res.status(500).json({
      HasError: true,
      Message: `Satış tamamlama hatası: ${errorMessage}`,
      Details: errorDetail
    });
  }
});

app.listen(PORT, () => {
  console.log(`Sunucu http://localhost:${PORT} adresinde çalışıyor`);
});