document.addEventListener('DOMContentLoaded', function () {
  const navLinks = document.querySelectorAll('.nav-link');
  const pages = document.querySelectorAll('.page');

  navLinks.forEach(link => {
    link.addEventListener('click', function (e) {
      e.preventDefault();
      const targetPage = this.getAttribute('data-page');

      navLinks.forEach(link => link.classList.remove('active'));
      this.classList.add('active');

      pages.forEach(page => {
        page.classList.remove('active');
        if (page.id === targetPage) {
          page.classList.add('active');
        }
      });

      if (targetPage === 'settings') {
        loadSettings();
        loadDevices();
      }
    });
  });

  let devices = JSON.parse(localStorage.getItem('pos_devices') || '[]');
  const deviceForm = document.getElementById('device-form');
  const deviceList = document.getElementById('device-list');
  const deviceNameInput = document.getElementById('device-name');
  const deviceSerialInput = document.getElementById('device-serial');
  const deviceIpInput = document.getElementById('device-ip');
  const deviceColorInput = document.getElementById('device-color');

  deviceForm.addEventListener('submit', function (e) {
    e.preventDefault();
    const deviceName = deviceNameInput.value.trim();
    const deviceSerial = deviceSerialInput.value.trim();
    const deviceIp = deviceIpInput.value.trim();
    const deviceColor = deviceColorInput.value;

    if (!deviceName || !deviceSerial || !deviceIp) {
      alert('Lütfen tüm cihaz bilgilerini doldurun');
      return;
    }

    const existingDevice = devices.find(device => device.serial === deviceSerial);
    if (existingDevice) {
      alert('Bu seri numarasıyla kayıtlı bir cihaz zaten var');
      return;
    }

    const newDevice = {
      id: generateDeviceId(),
      name: deviceName,
      serial: deviceSerial,
      ip: deviceIp,
      color: deviceColor,
      testStatus: null
    };
    devices.push(newDevice);
    saveDevices();
    loadDevices();
    testDeviceConnection(newDevice.id);

    deviceForm.reset();
    deviceColorInput.value = '#3498db';

    alert('Cihaz başarıyla eklendi');
  });

  function generateDeviceId() {
    return Date.now().toString(36) + Math.random().toString(36).substr(2, 5);
  }

  function saveDevices() {
    localStorage.setItem('pos_devices', JSON.stringify(devices));
  }

  function loadDevices() {
    deviceList.innerHTML = '';

    if (devices.length === 0) {
      deviceList.innerHTML = '<p class="no-device">Henüz cihaz eklenmemiş</p>';
      return;
    }

    devices.forEach(device => {
      const deviceElement = document.createElement('div');
      deviceElement.className = 'device-item';
      deviceElement.innerHTML = `
        <div class="device-color" style="background-color: ${device.color}"></div>
        <div class="device-info">
          <div class="device-name">${device.name}</div>
          <div class="device-details">
            <span>SN: ${device.serial}</span>
            <span>IP: ${device.ip}</span>
          </div>
          <div class="device-test-status ${device.testStatus ? (device.testStatus === 'success' ? 'success' : 'error') : ''}">
            ${device.testStatus ? (device.testStatus === 'success' ? 'Bağlantı Başarılı' : 'Bağlantı Hatası') : 'Test Edilmedi'}
          </div>
        </div>
        <div class="device-actions">
          <button class="device-test" data-id="${device.id}">Test Et</button>
          <button class="device-delete" data-id="${device.id}">Sil</button>
        </div>
      `;

      deviceList.appendChild(deviceElement);
    });

    const testButtons = document.querySelectorAll('.device-test');
    testButtons.forEach(button => {
      button.addEventListener('click', function () {
        const deviceId = this.getAttribute('data-id');
        testDeviceConnection(deviceId);
      });
    });

    const deleteButtons = document.querySelectorAll('.device-delete');
    deleteButtons.forEach(button => {
      button.addEventListener('click', function () {
        const deviceId = this.getAttribute('data-id');
        deleteDevice(deviceId);
      });
    });
  }

  function testDeviceConnection(deviceId) {
    const device = devices.find(d => d.id === deviceId);
    if (!device) return;

    const button = document.querySelector(`.device-test[data-id="${deviceId}"]`);
    if (button) {
      button.textContent = 'Test Ediliyor...';
      button.disabled = true;
    }

    fetch('/api/settings')
      .then(response => response.json())
      .then(settings => {
        const baseApiUrl = settings.apiUrl || '';
        const urlParts = baseApiUrl.split('://');
        if (urlParts.length < 2) {
          throw new Error('Geçersiz API URL formatı');
        }

        const protocol = urlParts[0];
        const pathParts = urlParts[1].split('/');
        pathParts[0] = device.ip;
        const testApiUrl = `${protocol}://${pathParts.join('/')}`;

        return fetch('/api/pairing', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json'
          },
          body: JSON.stringify({
            deviceSerial: device.serial,
            deviceIp: device.ip
          })
        });
      })
      .then(response => {
        if (!response.ok) {
          throw new Error(`Sunucu yanıtı: ${response.status}`);
        }
        return response.json();
      })
      .then(data => {
        if (data.HasError) {
          throw new Error(data.Message);
        }

        device.testStatus = 'success';
        saveDevices();
        loadDevices();
      })
      .catch(error => {
        console.error(`Cihaz testi hatası (${device.name}):`, error);

        device.testStatus = 'error';
        saveDevices();
        loadDevices();
      })
      .finally(() => {
        if (button) {
          button.textContent = 'Test Et';
          button.disabled = false;
        }
      });
  }

  function deleteDevice(deviceId) {
    if (!confirm('Bu cihazı silmek istediğinize emin misiniz?')) {
      return;
    }

    devices = devices.filter(device => device.id !== deviceId);
    saveDevices();
    loadDevices();
  }

  const paymentAmountInput = document.getElementById('payment-amount');
  const cashPaymentButton = document.getElementById('cash-payment');
  const cardPaymentButton = document.getElementById('card-payment');
  const statusMessage = document.getElementById('status-message');
  const deviceModal = document.getElementById('device-modal');
  const modalDeviceList = document.getElementById('modal-device-list');
  const closeModal = document.querySelector('.close-modal');

  closeModal.addEventListener('click', function () {
    deviceModal.style.display = 'none';
  });

  window.addEventListener('click', function (event) {
    if (event.target === deviceModal) {
      deviceModal.style.display = 'none';
    }
  });

  function fillModalDeviceList(paymentMethod) {
    modalDeviceList.innerHTML = '';

    const workingDevices = devices.filter(device => device.testStatus === 'success');

    if (workingDevices.length === 0) {
      modalDeviceList.innerHTML = `
        <p class="no-device">Bağlantı testi başarılı olan cihaz bulunamadı.</p>
        <p class="no-device-hint">Lütfen önce cihaz ekleyin ve bağlantı testini başarıyla tamamlayın.</p>
      `;
      return false;
    }

    workingDevices.forEach(device => {
      const deviceElement = document.createElement('div');
      deviceElement.className = 'modal-device-item';
      deviceElement.setAttribute('data-id', device.id);
      deviceElement.setAttribute('data-serial', device.serial);
      deviceElement.setAttribute('data-ip', device.ip);
      deviceElement.style.borderLeftColor = device.color;

      deviceElement.innerHTML = `
        <div class="modal-device-name">${device.name}</div>
        <div class="modal-device-details">
          <span>SN: ${device.serial}</span>
          <span>IP: ${device.ip}</span>
        </div>
      `;

      deviceElement.addEventListener('click', function () {
        const deviceSerial = this.getAttribute('data-serial');
        const deviceIp = this.getAttribute('data-ip');

        deviceModal.style.display = 'none';

        startPayment(deviceSerial, deviceIp, paymentMethod);
      });

      modalDeviceList.appendChild(deviceElement);
    });

    return true;
  }

  cashPaymentButton.addEventListener('click', function () {
    const amount = parseFloat(paymentAmountInput.value);
    if (isNaN(amount) || amount <= 0) {
      showStatusMessage('Lütfen geçerli bir tutar girin!', 'error');
      return;
    }

    if (fillModalDeviceList('cash')) {
      deviceModal.style.display = 'block';
    } else {
      showStatusMessage('Lütfen önce ayarlar sayfasından cihaz ekleyin ve bağlantı testini yapın!', 'error');
    }
  });

  cardPaymentButton.addEventListener('click', function () {
    const amount = parseFloat(paymentAmountInput.value);
    if (isNaN(amount) || amount <= 0) {
      showStatusMessage('Lütfen geçerli bir tutar girin!', 'error');
      return;
    }

    if (fillModalDeviceList('creditCard')) {
      deviceModal.style.display = 'block';
    } else {
      showStatusMessage('Lütfen önce ayarlar sayfasından cihaz ekleyin ve bağlantı testini yapın!', 'error');
    }
  });

  function startPayment(deviceSerial, deviceIp, paymentMethod) {

    disablePaymentButtons();

    const amount = parseFloat(paymentAmountInput.value);

    showStatusMessage(`${amount} ₺ ${paymentMethod === 'cash' ? 'nakit' : 'kredi kartı'} ödeme işlemi başlatılıyor...`, 'info');

    const sampleCart = [
      {
        id: "P001",
        name: "Ödeme",
        price: amount,
        quantity: 1
      }
    ];

    const paymentData = {
      cart: sampleCart,
      paymentMethod: paymentMethod,
      deviceSerial: deviceSerial,
      deviceIp: deviceIp
    };

    fetch('/api/complete-sale', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(paymentData)
    })
      .then(response => {
        if (!response.ok) {
          throw new Error(`Sunucu yanıtı: ${response.status}`);
        }
        return response.json();
      })
      .then(data => {
        if (data.HasError) {
          throw new Error(data.Message);
        }

        const deviceInfo = devices.find(device => device.serial === deviceSerial);
        const deviceName = deviceInfo ? deviceInfo.name : deviceSerial;

        showStatusMessage(`${amount} ₺ ${paymentMethod === 'cash' ? 'nakit' : 'kredi kartı'} ödeme "${deviceName}" cihazında başarıyla tamamlandı!`, 'success');
        enablePaymentButtons();
      })
      .catch(error => {
        showStatusMessage(`Ödeme hatası: ${error.message}`, 'error');
        console.error("Ödeme hatası detayı:", error);

        enablePaymentButtons();
      });
  }

  function disablePaymentButtons() {
    cashPaymentButton.disabled = true;
    cardPaymentButton.disabled = true;
    paymentAmountInput.disabled = true;

    cashPaymentButton.classList.add('disabled');
    cardPaymentButton.classList.add('disabled');
    paymentAmountInput.classList.add('disabled');
  }

  function enablePaymentButtons() {
    setTimeout(() => {
      cashPaymentButton.disabled = false;
      cardPaymentButton.disabled = false;
      paymentAmountInput.disabled = false;

      cashPaymentButton.classList.remove('disabled');
      cardPaymentButton.classList.remove('disabled');
      paymentAmountInput.classList.remove('disabled');
    }, 2000);
  }

  function showStatusMessage(message, type) {
    statusMessage.textContent = message;
    statusMessage.className = `status-message ${type}`;
    statusMessage.style.display = 'block';

    setTimeout(() => {
      statusMessage.style.display = 'none';
      statusMessage.className = 'status-message';
    }, 5000);
  }

  const settingsForm = document.getElementById('settings-form');
  const apiUrlInput = document.getElementById('api-url');

  function loadSettings() {
    fetch('/api/settings')
      .then(response => response.json())
      .then(data => {
        apiUrlInput.value = data.apiUrl || '';
      })
      .catch(error => {
        console.error('Ayarlar yüklenirken hata oluştu:', error);
      });
  }

  settingsForm.addEventListener('submit', function (e) {
    e.preventDefault();

    const apiUrl = apiUrlInput.value.trim();
    if (!apiUrl) {
      alert('Lütfen API URL\'sini girin');
      return;
    }

    fetch('/api/settings', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        apiUrl: apiUrl
      })
    })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          alert('Ayarlar başarıyla kaydedildi');

          devices.forEach(device => {
            device.testStatus = null;
          });
          saveDevices();
          loadDevices();
        } else {
          alert('Ayarlar kaydedilirken bir hata oluştu');
        }
      })
      .catch(error => {
        alert(`Hata: ${error.message}`);
      });
  });

  loadDevices();

  if (document.querySelector('.page.active').id === 'settings') {
    loadSettings();
  }
});